/////////////////////////////////////////////////////////////////////////////////
//
// Shader  : TheEmu - Tunnel A.fsh
// Creator : TheEmu
// Version : 1.2.2
// Date    : 2015/06/08
// License : Creative Commons Attribution-NonCommercial-ShareAlike 3.0 Unported License.
//
// History:
//   1.0   - 2015/05/12 - Initial release
//   1.0.1 - 2015/05/12 - Bug fix. Used abs(qq) in tunnelRadius.
//   1.1   - 2015/05/22 - Added fadeMode as a control parameter.
//   1.2   - 2015/06/06 - Added shapeFamily as a control parameter.
//   1.2.1 - 2015/06/07 - Added xyScale and xyScale2 control parameters.
//   1.2.2 - 2015/06/08 - Added normal/reversed texture index control.
//
/////////////////////////////////////////////////////////////////////////////////
//
// This shader was inspired by Inigo Quilez's Deform - Square Tunnel downloaded
// from ShaderToy.com,  but other than using the same Minkowski distance metric
// has nothing in common with it,  though  it  can be configured to produce the
// same tunnel shape.
//
// It is highly configurable, with the configuration parameters taking the form
// of a set of shader uniform inputs that may be defined in the scene file. The
// parameters control the tunnel shape, how its walls are tiled etc.
//
// With the single exception of the image that is used for the tunnel walls the
// control parameters are all optional  -  if they are not defined in the scene
// file they take the value 0 which is either the natural default value for the
// parameter or is treated as a special case indicating that some other default
// value is used.  All of the shader control parameters are described in detail
// in the following comments.
//
// Note, only the variables declared as uniform are inputs, the other variables
// declared with them are simply constants derived from them.
//
/////////////////////////////////////////////////////////////////////////////////
//
// Possible Enhancements
//
//    Bending the tiles
//    Curving the tunnel
//    Tile overlap with gradual fade in/fade out over the transition region
//    Better handling of tunnel end
//    Varying tunnel size
//    Better "perspective"
//    Separate specification of tunnel mouth
//    Maximum tunnel size and/or length
//    Other families of tunnel shapes
//
/////////////////////////////////////////////////////////////////////////////////

// Standard shader inputs.

uniform float u_Elapsed;    // The elapsed time in seconds
uniform vec2  u_WindowSize; // Window dimensions in pixels

   float wAspectRatio = u_WindowSize.x/u_WindowSize.y;

   vec2 wEqualizer = (wAspectRatio > 1.0) ? vec2( wAspectRatio, 1.0 )
                                       : vec2( 1.0, 1.0/wAspectRatio );

/////////////////////////////////////////////////////////////////////////////////

// Shader inputs that may be defined in the scene file.

// The image to be used for the tunnel walls is supplied using the following 2D
// sampler input channel. It may be any valid 2D sample,  e.g.  a  static image
// defined by a texture statement in the scene file, or a dynamic one generated
// in a framebuffer by a shader or it may be from a clip.  An image must always
// be provided.

uniform sampler2D iChannel0;

// By default, speed 0.0, the tunnel will be static.  Positive values for speed
// will generate the effect of motion into the tunnel and negative values  that
// of motion out of the tunnel, though using "unusual" values for other control
// parameters can reverse this.  If used speed should normally have an absolute
// value of less than 10, though higher speeds are permitted.

uniform float speed;

   float time = u_Elapsed * speed;

// The shader can operate in several modes controlled by the mode parameter.
//
// The wrapping modes fall into two groups, these being:-
//
//    Symetric 180 degree wrapping modes
//
//      00 - Tunnel is symetric about vertical axis
//      01 - Tunnel is symetric about horizontal axis
//
//      10 - As mode 0 but flipped about horizontal axis
//      11 - As mode 1 but flipped about vertical axis
//
//    360 wrapping modes starting at multiples of 45 degrees
//
//      20 to 27 - anticlockwise, second digit specifies starting segment
//      30 to 27 - clockwise, second digit specifies starting segment

uniform int mode;

// The image used for the tunnel is repeated as many times as is  necessary  to
// cover the full 360 degrees of the tunnel. By default the repeat angle is set
// by the mode parameter being 180 for modes 0..3 and 360 for modes 11..13, but
// this is can be overridden by specifying a non zero value for the repeatAngle
// which is the angle in degrees after which the image is repeated.

uniform float repeatAngle;

// The shape of the tunnel's cross section is controled by the shapeFamily  and
// shape parameters.  Of  these  shapeFamily selects which algorithm is used to
// determine the tunnel's "radius" and shape provides parameters that are  then
// used by that algorithm. For technical details of the algorithms refer to the
// comments for the tunnelRadius function in the body of this shader,  here  we
// only describe the general forms of tunnels that can be acheived and how they
// are affected the shape parameter.
//
// Currently there are only two shapeFamilies
//
//   shapeFamily 0, Ovals and rounded rectangles
//   shapeFamily 1, Sharp cornered rectangles and distorted versions of them
//
// In both cases the first component of shape controls the basic shape and the
// second its aspect ratio. The interpretation of the first parameter, but not
// the second, depends on the shape family.
//
// For shape family 0 if the first component of shape is 0.0 then the tunnel is
// eliptical, values greater than 0.0 correspond to shapes intermediate between
// elipses and rectangles with large values,  e.g. 10, produce tunnels that are
// rectangular with rounded corners. Negative values may also be used, but they
// are less useful with values between 0.0 and -0.5 producing shapes  that  are
// intermediatea between an elipse and a diamond, -0.5 a diamond,  between -0.5
// and -1.0 inward curving diamonds with a degenerate case  at  -1.0  when  the
// tunnel collapses to a pair of crossing line. Values more negative than -1.0
// produce hyberbollic sided tunnels.
//
// Note if too large an absolute value is used then the limitations of floating
// point arithmetic cause some problems resulting in the tunnel being truncated
// rather than disappering to infinity.  Values greater than 50 should probably
// be avoided.
//
// For shape family 1 if the first component of shape is 0.0 then the tunnel is
// rectangular with sharp corners.  Values  greater than 0.0 cause the sides to
// bulge outwards with the corner angles being wider than a right angle. If the
// value is much larger than 1.0,  e.g. 10.0,  the corners approach 180 degrees
// and the tunnel cross section becomes oval.  Small negative  values cause the
// sides to bulge inwards, with degenerate case occuring at -1.0 when the sides
// collapse into a pair of crossing lines.  Values  that ate more negative than
// -1.0 cause so much inwards bulging that the the bulges are larger  than  the
// radius of the tunnel - causing the tunnel cross section to have four lobes.
//
// The second part of shape always controls the aspect ratio of the tunnel i.e.
// its width to height ratio. However, an aspect ratio of 0.0 is a special case
// which results in a tunnel with the same aspect ratio as the window in  which
// the tunnel is to be rendered.
//
// Shape defaults to (0.0,0.0) resulting in either an eliptical or  rectangular
// tunnel,  depending on shapeFamily,  with the same aspect ratio as the window
// in which it is rendered.
//
// Note, the value used for the first component of shape can cause the apparent
// length of each tunnel section to change.  This can be compensated for use of
// the zScale control parameter described below.

uniform int shapeFamily;
uniform vec2 shape;

   float tunnelAspectRatio = (shape.y==0.0) ? wAspectRatio : shape.y;

// By default the tunnel's vanishing point will be the center of the window  in
// which it is being displayed, but this may be changed by spacifying that some
// other center is used.  The values for center take the form (dx,dy) where the
// dx component ranges from -1.0 at the left edge of the window to +1.0 at  the
// right edge and dy ranges from -1.0 at the bottom of the window  to  +1.0  at
// the top. Values outside of these ranges are valid but are not expected to be
// used as they would move the tunnel's center out of the window.

uniform vec2 center;

// By default the scaling is uniform,  i.e.  it is the same in both the x and y
// directions. This can be changed by specifying a non-zero xyScale. The effect
// of this scaling is slightly more subtle than it might seem from  the  simple
// description just given.  This is because scaling x and y in this way changes
// the values used to index into the texture image that tiles the tunnel walls.
// Possibly the most useful way in which this scaling can be used is to  adjust
// matters so that the floor, sides and ceiling parts of the texture are better
// fits to the floor, sides and ceiling of the tunnel.
//
// By default the xy scaling is uniform,  i.e.  the  same at the top and bottom
// and the same at both sides.  However,  this  can be modified by specifying a
// values for xyScale2, which defaults to xyScale.  The  actual values used for
// the scaling factors then vary linearly from xyScale at (x,y)=(-1.0,-1.0)  to
// xyScale2 at (x,y)=(1.0,1.0).
//
// The effect of using just xyScale is  quite  straightforward,  stretching  or
// squeezing the final image that of xyScale2 is not. For shapeFamily 0 it will
// cause the tunnel cross section to become distorted from an elipse to an  egg
// shape.  For shapeFamily 1 the effect is less pronounced being primarily just
// a vertical or horizontal stretching, but it can produce an apparent curve in
// the z direction (along the tunnel) at one edge of the image.
//
// Typical values for these parameters are in the  range  0.2 to 5.0,  but  any
// values can be used, even negative ones.

uniform vec2 xyScale;
uniform vec2 xyScale2;

   vec2 userScaling1 = (xyScale==vec2(0.0))  ? vec2(1.0,1.0) : 1.0/xyScale;
   vec2 userScaling2 = (xyScale2==vec2(0.0)) ? userScaling1  : 1.0/xyScale2;

// The generated tunnel will appear to be composed of  repetitions  of  tubular
// elements laid end to end along the Z axis, i.e. into the screen. The size of
// these elements may be controlled by specifying a non-zero value  for  zScale
// which will then be used to multiply the default size.  Any value may be used
// for zScale, though 0.0 is ignored (effectively being used as if it were 1.0)
// but typical values are expected to be between 0.5 and 5.0.

uniform float zScale;

   float zScaleFactor = (zScale==0.0) ? 1.0 : 1.0/zScale;

// By default the X and Y coordinates are horizontal and vertical, however they
// may be rotated by specifying a non zero rotation angle, alpha, in degrees. A
// positive value for alpha signifies a clockwise rotation of the tunnel.

uniform float alpha;

   float sinAlpha = sin(radians(alpha));
   float cosAlpha = cos(radians(alpha));

   mat2 rotateAlpha = mat2(cosAlpha,sinAlpha,-sinAlpha,cosAlpha);

// By default there is no fading as the distance along  the  tunnel  increases,
// however this can be overriden by specifying a non-zero value for  fade.  Any
// value can be used but typical values are expected to be in the range 0.0  to
// 5.0 as larger values will blank out most of the tunnel.

uniform float fade;

// By default any fading will be towards black but this can be overriden by the
// use of fadeColour. The three components of which are the red, green and blue
// intensities of the colour to fade towards.  Fading towards black will simply
// darken the tunnel while fading toward white makes the tunnel misty.

uniform vec3 fadeColour;

   vec4 fadeColourV4 = vec4(fadeColour,1.0);

// By default the colour fading takes no account of zScale  or  of  transparent
// tunnel wall areas.  This  can lead to a noticable sudden start to the fading
// some distance down the tunnel and unwanted obscuring of the background  seen
// through transparent wall areas.  This can be controled using fadeMode  which
// is a three digit integer, abc, where
//
//    a = 0  -  fading clamped - no area will be washed out
//    a = 1  -  fading not clamped - colour washout may occur
//
//    b = 0  -  transparent areas faded - background will be faded
//    b = 1  -  transparent areas not faded - background not faded
//
//    c = 0  -  zScale ignored - fading may start part way down tunnel
//    c = 1  -  zScale compensation applied - all of tunnel affected
//
// If fading is not clamped then areas near the front of the tunnel may  suffer
// from a negative shading effect.  In  the normal case of "fade to black" this
// means that the front areas may become washed out. This effect should only be
// noticable is zScale is greater than 1.0 when it can result in the  edges  of
// the tunnel's image looking somewhat bleached.

uniform int fadeMode;

// By default the opacity of the tunnel will match that of the  image  used  to
// provide the texture its walls. However, this may be overridden by specifying
// a non-zero value for translucency. Translucency values are clamped to lie in
// the range 0.0 to 1.0 with the tunnel walls becoming more transparent as  the
// value is increased  -  with 1.0 being fully transparent and hence invisible.
// Note,  however,  that any fully transparent region in the image used for the
// tunnel walls texture will always remain fully transparent.

uniform float translucency;

   float opacity = 1.0 - clamp(translucency,0.0,1.0);

// By default each copy of the image used for the walls will be oriented in the
// same way.  Unless the image has been specialy designed to avoid it this will
// cause clearly visible discontinuities where the image tiles meet.  This  can
// be aleviated by flipping the alternate tiles and this is controlled  by  the
// tileScheme parameter.  This is a 3 digit integer where the least significant
// two digits control tile flpping and the most significant digit controls  how
// the tile's texture is indexed.
//
// Of the two digits controling tile flipping the most significant controls the
// flipping in the y direction, i.e. verticaly, and the least significant digit
// controls flipping in the z direction, i.e. along the tunnel.  In  both cases
// the interpretation of the digit is
//
//     0 - no tile flipping
//     1 - flip even numbered tiles
//     2 - flip odd numbered tiles
//
// The most significant digit of the three digits for tileScheme  controls  how
// the tile's texture is indexed according to the following list
//
//     0 - normal              - index by z,alpha
//     1 - reversed x          - index by 1-z,alpha
//     2 - reversed y          - index by z,1-alpha
//     3 - reversed x and y    - index by 1-z,1-alpha
//     4 - swapped             - index by alpha,z
//     5 - swapped+reversed x  - index by 1-alpha,z
//     6 - swapped+reversed y  - index by alpha,1-z
//     7 - swapped+reversed xy - index by 1-alpha,1-z
//
// where z and alpha are the fractional parts of the distance along the  tunnel
// and the angle measured around the tunnel's axis.

uniform int tileScheme;

// By default the tunnel walls are tiled using a simple rectangular pattern  of
// repetitions of the tunnel wall image with one axis parallel to that  of  the
// of the tunnel.  However,  this pattern may be modified such that the pattern
// twists around the tunnel axis. This is controlled by the following parameter
// which works by changing the effective shape of the tunnel wall image from  a
// rectangles to a diamond using the transformation
//
//      (X,Y) => ( X + A*Y, Y + B*X )
//
// where A and B are constants given by the two components of twist.
//
// Depending on the values of A and B a variety of effects can be obtained,  in
// the following it is assumed that a 360 degree wrapping mode is used.
//
// Non-zero values for A cause lines that were perpendicular to the axis of the
// tunnel,  i.e. the closed loops made by vertical sections through the tunnel,
// to spiral about the axis. In general this will introduce discontinuities and
// a closed loop will become a spiral segement, but if A*N is an integer, where
// N is 360/repeatAngle,  then the end of one segment will connect to the start
// of the next giving a continuous spiral down the tunnel.
//
// Non-zero values for B cause lines that were parallel to the tunnel's axis to
// spiral about that axis leaving lines at right angles to the axis  unchanged.
// In this case all values lead to continuous spirals.
//
// The simplest spirals occur if either A or B is zero.  If  both  are non-zero
// then more complex twisting of the pattern will result. A special case occurs
// when A and B are both 1.0 as the transformation given above the reduces to
//
//        (X,Y) => ( X + Y, X + Y )
//
// and instead of transforming the image to a diamond it becomes transformed to
// a single line. In this case the tiles are one dimensional and the end result
// is the tunnel walls are tiled with ribbons.
//
// Values for A and B are expected to be in the range -5.0 to +0.5, values that
// lie outside this range are valid but the resulting spirals are rather tight.

uniform vec2 twist;

// By default the tunnel is endless,  however this can be changed by specifying
// a positive value for endSize.  Assuming that all other parameters have their
// default values setting endSize to 1.0 will cause the end of  the  tunnel  to
// just reach the edges of the window and smaller values cause it to reduce  in
// size, and hence retreat down the tunnel into the distance.

uniform float endSize;

// By default the end of the tunnel is transparent,  but any colour may be used
// by setting endColour. The components are the Red, Green and Blue intensities
// and the opacity. It will have no effect if the endsize is 0.0 or less.

uniform vec4 endColour;

// The tunnel can be regarded as a series of tubes aligned along the Z axis. By
// default there is no gap btween the tubes and the tunnel is continuous.  This
// may however be overrriden by providing a non zero value  for  gapSize  which
// specifies the relative size of the inter-tube gaps. Values for gapSize ought
// to lie between 0.0 and 1.0 corresponding to no gap and all gap respectively.

uniform float gapSize;

   float segmentSize = 1.0 - clamp(gapSize,0.0,1.0);

// By default the gaps in the tunnel is transparent, but any colour may be used
// by setting gapColour. The components are the Red, Green and Blue intensities
// and the opacity. It will have no effect if the gapsize is 0.0 or less.

uniform vec4 gapColour;

/////////////////////////////////////////////////////////////////////////////////

// TunnelCoordinates determines the (X,Y) coordinates of the current pixel. The
// coordinates origin is at the center of the tunnel.

vec2 TunnelCoordinates ( void )
 {

   // Rescale so that all the range of X and Y are both from -1 to +1.

   vec2 uv = ( 2.0*gl_FragCoord.xy/u_WindowSize.xy - 1.0 );

   // Move the origin to be the center of the tunnel.

   uv = uv - center;

   // Rescale so that X and Y have equal scales and apply any user
   // defined scaling factor. The user scaling factors are go from
   // userScaling1 at uv=(-1,-1) to userScaling2 at uv=(1,1).

   uv = uv * wEqualizer * mix ( userScaling1, userScaling2, (1.0+uv)/2.0 );

   // Rotate by alpha.

   return rotateAlpha * uv;

 }

/////////////////////////////////////////////////////////////////////////////////

// TunnelTheta calculates the angle round the tunnel that  corresponds  to  the
// current pixel and rescales it such that 1.0 corresponds to the repeat  angle
// used for the tunnel wall tiling.  The modes all reduce to one of two general
// cases which are handled by their own functions.

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

// TunnelThetaA handles the simple 180 degree wrapping modes.

float TunnelThetaA ( vec2 uv, float a, float b )
 {
   return a*radians(180.0) + b*abs(atan(uv.y,uv.x));
 }

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

// TunnelThetaB handles the oblique 180 degree wrapping modes.

float TunnelThetaB ( vec2 uv, float a, float b )
 {
   float theta = atan(a*uv.y,a*uv.x) + b*radians(360.0+45.0);
   theta = mod(theta,radians(360.0));
   if ( theta > radians(180.0) ) theta = theta - radians(360.0);
   return abs(theta);
 }

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

// TunnelThetaC handles the 360 degree wrapping modes.

float TunnelThetaC ( vec2 uv, int n, float a )
 {
   return radians(45.0*n) + a*atan(uv.y,uv.x);
 }

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

float TunnelTheta ( vec2 uv )
 {
   float theta;

   // Determine which of the mode handlers to call and then call it
   // with the appropriate set mode specific parameter values.

   int m = mode;
   if ( m > 19 ) m = m - int(mod(m,10));

   switch ( m )
    {
      case 00 : theta = TunnelThetaA ( uv.yx,  0.0, +1.0 ); break;
      case 01 : theta = TunnelThetaA ( uv.xy,  0.0, +1.0 ); break;
      case 10 : theta = TunnelThetaA ( uv.yx, +1.0, -1.0 ); break;
      case 11 : theta = TunnelThetaA ( uv.xy, +1.0, -1.0 ); break;

      case 02 : theta = TunnelThetaB ( uv.yx, +1.0, -1.0 ); break;
      case 03 : theta = TunnelThetaB ( uv.xy, -1.0, +1.0 ); break;
      case 12 : theta = TunnelThetaB ( uv.xy, +1.0, +1.0 ); break;
      case 13 : theta = TunnelThetaB ( uv.yx, -1.0, -1.0 ); break;

      case 20 : theta = TunnelThetaC ( uv, mode-20, +1.0 ); break;
      case 30 : theta = TunnelThetaC ( uv, mode-30, -1.0 ); break;
    }

   // Get the wall tiling repeat angle replacing any defaulted, i.e
   // zero, value with the appropriate mode dependant default.

   float rptAngle = repeatAngle;

   if ( rptAngle == 0.0 )
    {  rptAngle = (mode<20) ? 180.0 : 360.0;
    }

   // Return the rescaled theta angle.

   return theta/radians(rptAngle);

 }

/////////////////////////////////////////////////////////////////////////////////

// Tunnel radius calculates the on screen radius of the tunnel corresponding to
// the current pixel, i.e. its distance from the center of the tunnel.  However
// a modified metric is used for the radius which has the effect of  distorting
// the tunnel according the shape parameter.
//
// For shapeFamily = 0, which is the default,  the  tunnel radius is calculated
// using a Minkowski metric, r = ( x^p + y^p ) ^ (1/p)  where a ^ b is a raised
// to the power b - this reduces to the normal Euclidian metric when p = 2 when
// the preceeding formula is just Pythagoras' theorum.
//
// For shapeFamily 1 the tunnel radius is calculated by  a  modified  Chebyshev
// metric.  The  normal  Chebyshev metric is the limiting case of the Minkowski
// metric where p is infinite but is more easily calculated using r = min(x,y).
// The modified version used here adds a second term, length(x,y)*f, which  has
// no effect if f is zero.

float TunnelRadius ( vec2 uv )
 {
   float rr = 1.0; // Used if metric is invalid

   vec2 qq = uv * vec2(1.0,tunnelAspectRatio);

   switch ( shapeFamily )
    {
      case 0: // Minkowski metric

         float p = ( 1.0 + shape.x ) * 2.0;

         qq = abs(qq);

         qq.x = pow ( qq.x, p );
         qq.y = pow ( qq.y, p );

         rr = pow ( qq.x + qq.y, 1.0/p );

         break;

      case 1: // Modified Chebyshev metric

         qq = abs ( qq );
         rr = max ( qq.x, qq.y );

         rr = abs ( rr + length(qq)*shape.x );

         break;

    }

   return rr;
 }

/////////////////////////////////////////////////////////////////////////////////

// TunnelWall determines the colour, including the alpha component, to be used
// for the current tunnel wall pixel.

vec4 TunnelWall ( float r, float theta )
 {
   // Tunnel wall coordinates to be used to index into the  tunnel
   // wall txture.  qq.x is measured along along the tunnel's axis
   // and qq.y is the the angle round that axis. These coordinates
   // may be optionly be twisted and stretched.

   vec2 qq = vec2 ( 1.0/(r+0.01) + time, theta );

   qq = qq + twist * qq.yx;
   qq.x = qq.x * zScaleFactor;

   // Optionaly flip the texture indices to eliminate first  order
   // discontinuities between neighbouring tunnel patches. The two
   // digit integer tileScheme uniform controls this process.

   int kx = int(floor(qq.x)); kx = kx - 2*(kx/2);
   int ky = int(floor(qq.y)); ky = ky - 2*(ky/2);

   int flip = (tileScheme<100) ? tileScheme : tileScheme - 100;

   switch ( flip )
    {
      case 0: break;

      case 1: if ( kx == 0 ) qq.x = 1.0 - qq.x; break;

      case 2: if ( kx != 0 ) qq.x = 1.0 - qq.x; break;

      case 10: if ( ky == 0 ) qq.y = 1.0 - qq.y; break;

      case 11: if ( ky == 0 ) qq.y = 1.0 - qq.y;
               if ( kx == 0 ) qq.x = 1.0 - qq.x; break;

      case 12: if ( ky == 0 ) qq.y = 1.0 - qq.y;
               if ( kx != 0 ) qq.x = 1.0 - qq.x; break;

      case 20: if ( ky != 0 ) qq.y = 1.0 - qq.y; break;

      case 21: if ( ky != 0 ) qq.y = 1.0 - qq.y;
               if ( kx == 0 ) qq.x = 1.0 - qq.x; break;

      case 22: if ( ky != 0 ) qq.y = 1.0 - qq.y;
               if ( kx != 0 ) qq.x = 1.0 - qq.x; break;

    }

   qq = fract(qq);

   switch ( tileScheme / 100 )
    {
      case 0:                           break;
      case 1: qq.x  = 1.0 - qq.x;       break;
      case 2: qq.y  = 1.0 - qq.y;       break;
      case 3: qq.xy = 1.0 - qq.xy;      break;
      case 4: qq.xy = qq.yx;            break;
      case 5: qq = vec2(1.0-qq.y,qq.x); break;
      case 6: qq = vec2(qq.y,1.0-qq.x); break;
      case 7: qq = 1.0 - qq.yx;         break;
    }

   // Get the pixel's colour and apply translucency.  Note that any
   // transparent region in the tunnel's source texture will remain
   // transparent thereby allowing any  background  to  be  visible
   // through the windows that they provide.

   vec4 colour;

   if ( qq.x > segmentSize )
    {
      colour = gapColour;
    }
   else
    {
      qq.x = qq.x / segmentSize;
      colour = texture2D ( iChannel0, qq );
      colour.a = colour.a * opacity;
     }

   // Apply any fading.

   float f = 0.0;
   float q = r*zScaleFactor;

   switch ( fadeMode )
    { case 00: case 100 : f = fade; q = r;                           break;
      case 01: case 101 : f = fade;                                  break;
      case 10: case 110 : if ( colour.a > 0.0 ) { f = fade; q = r; } break;
      case 11: case 111 : if ( colour.a > 0.0 ) { f = fade; }        break;
    }

   f = pow(q,f);

   if ( fadeMode < 100 )
    { f = clamp(f,0.0,1.0);
    }

   return mix ( fadeColourV4, colour, f );

 }

/////////////////////////////////////////////////////////////////////////////////

void main ( void )
 {
   vec2 uv = TunnelCoordinates();
   float r = TunnelRadius ( uv );

   gl_FragColor = (r<endSize) ? endColour
                              : TunnelWall ( r, TunnelTheta(uv) );

 }

/////////////////////////////////////////////////////////////////////////////////
